<?php
namespace WPO\IPS\Settings;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if ( ! class_exists( '\\WPO\\IPS\\Settings\\SettingsGeneral' ) ) :

class SettingsGeneral {

	protected $option_name = 'wpo_wcpdf_settings_general';

	protected static $_instance = null;

	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	public function __construct()	{
		add_action( 'admin_init', array( $this, 'init_settings' ) );
		add_action( 'wpo_wcpdf_settings_output_general', array( $this, 'output' ), 10, 2 );
		add_action( 'wpo_wcpdf_before_settings', array( $this, 'attachment_settings_hint' ), 10, 2 );
		add_action( 'wp_ajax_wcpdf_get_country_states', array( $this, 'ajax_get_shop_country_states' ) );

		// Display an admin notice if shop address fields are empty.
		add_action( 'admin_notices', array( $this, 'display_admin_notice_for_shop_address' ) );
	}

	public function output( $section, $nonce ) {
		if ( ! wp_verify_nonce( $nonce, 'wp_wcpdf_settings_page_nonce' ) ) {
			return;
		}

		settings_fields( $this->option_name );
		do_settings_sections( $this->option_name );

		submit_button();
	}

	public function init_settings() {
		$page = $option_group = $option_name = $this->option_name;

		$template_base_path     = ( defined( 'WC_TEMPLATE_PATH' ) ? WC_TEMPLATE_PATH : $GLOBALS['woocommerce']->template_url );
		$theme_template_path    = get_stylesheet_directory() . '/' . $template_base_path;
		$wp_content_dir         = defined( 'WP_CONTENT_DIR' ) && ! empty( WP_CONTENT_DIR ) ? str_replace( ABSPATH, '', WP_CONTENT_DIR ) : '';
		$theme_template_path    = substr( $theme_template_path, strpos( $theme_template_path, $wp_content_dir ) ) . 'pdf/yourtemplate';
		$plugin_template_path   = "{$wp_content_dir}/plugins/woocommerce-pdf-invoices-packing-slips/templates/Simple";
		$requires_pro           = function_exists( 'WPO_WCPDF_Pro' ) ? '' : sprintf( /* translators: 1. open anchor tag, 2. close anchor tag */ __( 'Requires the %1$sProfessional extension%2$s.', 'woocommerce-pdf-invoices-packing-slips' ), '<a href="' . esc_url( admin_url( 'admin.php?page=wpo_wcpdf_options_page&tab=upgrade' ) ) . '">', '</a>' );
		$states                 = wpo_wcpdf_get_country_states( $this->get_setting( 'shop_address_country' ) );
		$missing_template_files = $this->get_missing_template_files();
		$has_vat_plugin_active  = \wpo_ips_has_vat_plugin_active();
		$vat_plugin_notice      = '';

		if ( $has_vat_plugin_active ) {
			$vat_plugin_notice = '<div class="notice notice-info inline notice-wpo"><p>'
				. esc_html__( 'A VAT plugin is currently active. This option is disabled to avoid conflicts and duplicate VAT fields at checkout.', 'woocommerce-pdf-invoices-packing-slips' )
				. '</p></div>';
		}

		$settings_fields = array(
			array(
				'type'     => 'section',
				'id'       => 'general_settings',
				'title'    => __( 'General settings', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'section',
			),
			array(
				'type'     => 'setting',
				'id'       => 'download_display',
				'title'    => __( 'How do you want to view the PDF?', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'select',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'download_display',
					'options'     => array(
						'download' => __( 'Download the PDF' , 'woocommerce-pdf-invoices-packing-slips' ),
						'display'  => __( 'Open the PDF in a new browser tab/window' , 'woocommerce-pdf-invoices-packing-slips' ),
					),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'template_path',
				'title'    => __( 'Choose a template', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'select',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'      => $option_name,
					'id'               => 'template_path',
					'options_callback' => array( WPO_WCPDF()->settings, 'get_installed_templates_list' ),
					'description'      => sprintf(
						/* translators: 1: plugin template path, 2: theme template path */
						_n(
							'Want to use your own template? Copy the file from %1$s to your (child) theme in %2$s to customize it.',
							'Want to use your own template? Copy all the files from %1$s to your (child) theme in %2$s to customize them.',
							count( $missing_template_files ),
							'woocommerce-pdf-invoices-packing-slips'
						),
						'<code>' . esc_html( $plugin_template_path ) . '</code>',
						'<code>' . esc_html( $theme_template_path ) . '</code>'
					) . $this->render_missing_template_files_notice( $missing_template_files ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'template_ink_saving',
				'title'    => __( 'Ink saving mode', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'checkbox',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'       => $option_name,
					'id'                => 'template_ink_saving',
					'description'       => __( 'Apply ink-saving styles for this template, replacing dark backgrounds and colors with lighter alternatives.', 'woocommerce-pdf-invoices-packing-slips' ),
					'custom_attributes' => array(
						'data-show_for_option_name'   => $option_name . '[template_path]',
						'data-show_for_option_values' => wp_json_encode( apply_filters( 'wpo_ips_ink_saving_supported_templates', array( 'default/Simple' ) ) ),
					),
				),
			),
			array(
				'type'     => 'setting',
				'id'       => 'paper_size',
				'title'    => __( 'Paper size', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'select',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'paper_size',
					'options'     => apply_filters( 'wpo_wcpdf_template_settings_paper_size', array(
						'a4'     => __( 'A4' , 'woocommerce-pdf-invoices-packing-slips' ),
						'letter' => __( 'Letter' , 'woocommerce-pdf-invoices-packing-slips' ),
					) ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'test_mode',
				'title'    => __( 'Test mode', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'checkbox',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'test_mode',
					'description' => sprintf(
						'%1$s<br><strong>%2$s</strong><br><a href="%3$s" target="_blank">%4$s</a>',
						__( 'With test mode enabled, any document generated will always use the latest settings, rather than using the settings as configured at the time the document was first created.', 'woocommerce-pdf-invoices-packing-slips' ),
						__( 'Note: Invoice numbers and dates are not affected by this setting and will still be generated.', 'woocommerce-pdf-invoices-packing-slips' ),
						'https://docs.wpovernight.com/woocommerce-pdf-invoices-packing-slips/show-pdf-documents-with-the-latest-settings/',
						__( 'Learn more about test mode', 'woocommerce-pdf-invoices-packing-slips' )
					),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'currency_font',
				'title'    => __( 'Extended currency symbol support', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'checkbox',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'currency_font',
					'description' => __( 'Enable this if your currency symbol is not displaying properly' , 'woocommerce-pdf-invoices-packing-slips' ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'font_subsetting',
				'title'    => __( 'Enable font subsetting', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'checkbox',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'font_subsetting',
					'description' => __( "Font subsetting can reduce file size by only including the characters that are used in the PDF, but limits the ability to edit PDF files later. Recommended if you're using an Asian font." , 'woocommerce-pdf-invoices-packing-slips' ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'header_logo',
				'title'    => __( 'Shop header/logo', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'media_upload',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'          => $option_name,
					'id'                   => 'header_logo',
					'uploader_title'       => __( 'Select or upload your invoice header/logo', 'woocommerce-pdf-invoices-packing-slips' ),
					'uploader_button_text' => __( 'Set image', 'woocommerce-pdf-invoices-packing-slips' ),
					'remove_button_text'   => __( 'Remove image', 'woocommerce-pdf-invoices-packing-slips' ),
					'translatable'         => true,
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'header_logo_height',
				'title'    => __( 'Logo height', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'header_logo_height',
					'size'        => '5',
					'placeholder' => '3cm',
					'description' => __( 'Enter the total height of the logo in mm, cm or in and use a dot for decimals.<br/>For example: 1.15in or 40mm', 'woocommerce-pdf-invoices-packing-slips' ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'shop_name',
				'title'    => __( 'Shop Name', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'  => $option_name,
					'id'           => 'shop_name',
					'translatable' => true,
					'description'  => __( 'The name of your business or shop.', 'woocommerce-pdf-invoices-packing-slips' ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'vat_number',
				'title'    => __( 'Shop VAT Number', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'vat_number',
					'description' => __( 'You can display this number on the invoice from the document settings.', 'woocommerce-pdf-invoices-packing-slips' ) . ' ' . $requires_pro,
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'coc_number',
				'title'    => __( 'Shop Chamber of Commerce Number', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'coc_number',
					'description' => __( 'You can display this number on the invoice from the document settings.', 'woocommerce-pdf-invoices-packing-slips' ) . ' ' . $requires_pro,
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'shop_phone_number',
				'title'    => __( 'Shop Phone Number', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'  => $option_name,
					'id'           => 'shop_phone_number',
					'translatable' => true,
					'description'  => __( 'The phone number for your business location.', 'woocommerce-pdf-invoices-packing-slips' ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'shop_email_address',
				'title'    => __( 'Shop Email Address', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'  => $option_name,
					'id'           => 'shop_email_address',
					'translatable' => true,
					'type'         => 'email',
					'description'  => __( 'The email address for your business location.', 'woocommerce-pdf-invoices-packing-slips' ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'shop_address_line_1',
				'title'    => __( 'Shop Address Line 1', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'   => $option_name,
					'id'            => 'shop_address_line_1',
					'translatable'  => true,
					'description'   => __( 'The street address for your business location.', 'woocommerce-pdf-invoices-packing-slips' ),
					'action_button' => array(
						'class' => 'button sync-address',
						'icon'  => 'update',
						'title' => __( 'Sync with WooCommerce address', 'woocommerce-pdf-invoices-packing-slips' ),
						'text'  => '',
					),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'shop_address_line_2',
				'title'    => __( 'Shop Address Line 2', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'   => $option_name,
					'id'            => 'shop_address_line_2',
					'translatable'  => true,
					'description'   => __( 'An additional, optional address line for your business location.', 'woocommerce-pdf-invoices-packing-slips' ),
					'action_button' => array(
						'class' => 'button sync-address',
						'icon'  => 'update',
						'title' => __( 'Sync with WooCommerce address', 'woocommerce-pdf-invoices-packing-slips' ),
						'text'  => '',
					),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'shop_address_country',
				'title'    => __( 'Shop Country', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'select',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'   => $option_name,
					'options'       => array( '' => __( 'Select a country', 'woocommerce-pdf-invoices-packing-slips' ) ) + \WC()->countries->get_countries(),
					'id'            => 'shop_address_country',
					'translatable'  => true,
					'description'   => __( 'The country in which your business is located.', 'woocommerce-pdf-invoices-packing-slips' ),
					'action_button' => array(
						'class' => 'button sync-address',
						'icon'  => 'update',
						'title' => __( 'Sync with WooCommerce address', 'woocommerce-pdf-invoices-packing-slips' ),
						'text'  => '',
					),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'shop_address_state',
				'title'    => __( 'Shop State', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'select',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'   => $option_name,
					'options'       => $states ?: array( '' => __( 'No states available', 'woocommerce-pdf-invoices-packing-slips' ) ),
					'id'            => 'shop_address_state',
					'translatable'  => true,
					'description'   => __( 'The state in which your business is located.', 'woocommerce-pdf-invoices-packing-slips' ),
					'disabled'      => empty( $states ),
					'action_button' => array(
						'class' => 'button sync-address',
						'icon'  => 'update',
						'title' => __( 'Sync with WooCommerce address', 'woocommerce-pdf-invoices-packing-slips' ),
						'text'  => '',
					),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'shop_address_city',
				'title'    => __( 'Shop City', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'   => $option_name,
					'id'            => 'shop_address_city',
					'translatable'  => true,
					'description'   => __( 'The city in which your business is located.', 'woocommerce-pdf-invoices-packing-slips' ),
					'action_button' => array(
						'class' => 'button sync-address',
						'icon'  => 'update',
						'title' => __( 'Sync with WooCommerce address', 'woocommerce-pdf-invoices-packing-slips' ),
						'text'  => '',
					),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'shop_address_postcode',
				'title'    => __( 'Shop Postcode / ZIP', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'   => $option_name,
					'id'            => 'shop_address_postcode',
					'translatable'  => true,
					'description'   => __( 'The postal code, if any, in which your business is located.', 'woocommerce-pdf-invoices-packing-slips' ),
					'action_button' => array(
						'class' => 'button sync-address',
						'icon'  => 'update',
						'title' => __( 'Sync with WooCommerce address', 'woocommerce-pdf-invoices-packing-slips' ),
						'text'  => '',
					),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'shop_address_additional',
				'title'    => __( 'Shop Additional Info', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'textarea',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'  => $option_name,
					'id'           => 'shop_address_additional',
					'width'        => '72',
					'height'       => '8',
					'translatable' => true,
					'description'  => __( 'Any additional info about your business location.', 'woocommerce-pdf-invoices-packing-slips' ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'footer',
				'title'    => __( 'Footer: terms & conditions, policies, etc.', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'textarea',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name'  => $option_name,
					'id'           => 'footer',
					'width'        => '72',
					'height'       => '4',
					'translatable' => true,
				)
			),
			array(
				'type'     => 'section',
				'id'       => 'extra_template_fields',
				'title'    => __( 'Extra template fields', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'custom_fields_section',
			),
			array(
				'type'     => 'setting',
				'id'       => 'extra_1',
				'title'    => __( 'Extra field 1', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'textarea',
				'section'  => 'extra_template_fields',
				'args'     => array(
					'option_name'  => $option_name,
					'id'           => 'extra_1',
					'width'        => '72',
					'height'       => '8',
					'description'  => __( 'This is footer column 1 in the <i>Modern (Premium)</i> template', 'woocommerce-pdf-invoices-packing-slips' ),
					'translatable' => true,
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'extra_2',
				'title'    => __( 'Extra field 2', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'textarea',
				'section'  => 'extra_template_fields',
				'args'     => array(
					'option_name'  => $option_name,
					'id'           => 'extra_2',
					'width'        => '72',
					'height'       => '8',
					'description'  => __( 'This is footer column 2 in the <i>Modern (Premium)</i> template', 'woocommerce-pdf-invoices-packing-slips' ),
					'translatable' => true,
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'extra_3',
				'title'    => __( 'Extra field 3', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'textarea',
				'section'  => 'extra_template_fields',
				'args'     => array(
					'option_name'  => $option_name,
					'id'           => 'extra_3',
					'width'        => '72',
					'height'       => '8',
					'description'  => __( 'This is footer column 3 in the <i>Modern (Premium)</i> template', 'woocommerce-pdf-invoices-packing-slips' ),
					'translatable' => true,
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'checkout_field_enable',
				'title'    => __( 'Enable', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'checkbox',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'checkout_field_enable',
					'description' => __( 'Enable an optional custom field at checkout to collect customer identification data, such as tax IDs, company registration numbers, purchase order numbers, or internal customer references.', 'woocommerce-pdf-invoices-packing-slips' ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'checkout_field_label',
				'title'    => __( 'Label', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'text_input',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'checkout_field_label',
					'default'     => __( 'Customer identification', 'woocommerce-pdf-invoices-packing-slips' ),
					'description' => __( 'The label for the optional custom field displayed at checkout.', 'woocommerce-pdf-invoices-packing-slips' ),
				)
			),
			array(
				'type'     => 'setting',
				'id'       => 'checkout_field_as_vat_number',
				'title'    => __( 'Treat as VAT number', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'checkbox',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'checkout_field_as_vat_number',
					'disabled'    => $has_vat_plugin_active,
					'value'       => $has_vat_plugin_active ? false : $this->get_setting( 'checkout_field_as_vat_number' ),
					'description' => sprintf(
						/* translators: %s: WooCommerce EU VAT Compliance plugin link */
						__( 'When enabled, the checkout field is treated as a VAT number and may be used for basic VAT-related logic. Avoid enabling this option if you are already using a third-party VAT plugin, as it may result in duplicate or conflicting VAT fields. For advanced VAT validation, reporting, and full compliance with EU VAT rules, we recommend using %s.', 'woocommerce-pdf-invoices-packing-slips' ),
						'<a href="https://wpovernight.com/downloads/woocommerce-eu-vat-compliance/?utm_medium=plugin&utm_source=ips&utm_campaign=general-tab&utm_content=woocommerce-eu-vat-compliance-cross" target="_blank" rel="noopener noreferrer">WooCommerce EU VAT Compliance</a>',
					) . $vat_plugin_notice,
				),
			),
			array(
				'type'     => 'setting',
				'id'       => 'checkout_field_enable_my_account',
				'title'    => __( 'Editable in My Account', 'woocommerce-pdf-invoices-packing-slips' ),
				'callback' => 'checkbox',
				'section'  => 'general_settings',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'checkout_field_enable_my_account',
					'description' => __( 'Allow customers to edit the custom checkout field on their account details page. The value is saved to the customer profile and used for future checkouts only.', 'woocommerce-pdf-invoices-packing-slips' ),
				),
			),
		);

		if ( ! function_exists( 'WPO_WCPDF_Pro' ) ) {
			$settings_fields[] = array(
				'type'     => 'setting',
				'id'       => 'pro_notice_address_customization',
				'title'    => '',
				'callback' => 'html_section',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'pro_notice_address_customization',
					'class'       => 'wpo_wcpdf_pro_notice',
					'html'        =>
						__( 'Customize the formatting of shipping and billing addresses in PDF documents, and add custom fields as needed.', 'woocommerce-pdf-invoices-packing-slips' ) . ' ' .
						'<a href="' . esc_url( admin_url( 'admin.php?page=wpo_wcpdf_options_page&tab=upgrade' ) ) . '">' . __( 'Upgrade today!', 'woocommerce-pdf-invoices-packing-slips' ) . '</a>'
				),
			);

			$settings_fields[] = array(
				'type'     => 'setting',
				'id'       => 'pro_notice_multilingual',
				'title'    => '',
				'callback' => 'html_section',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'pro_notice_multilingual',
					'class'       => 'wpo_wcpdf_pro_notice',
					'html'        => __( 'Create PDF documents in multiple languages based on the customer\'s language or locale.', 'woocommerce-pdf-invoices-packing-slips' ) . ' ' .
						'<a href="' . esc_url( admin_url( 'admin.php?page=wpo_wcpdf_options_page&tab=upgrade' ) ) . '">' . __( 'Upgrade today!', 'woocommerce-pdf-invoices-packing-slips' ) . '</a>'
				),
			);

			$settings_fields[] = array(
				'type'     => 'setting',
				'id'       => 'pro_notice_static_files',
				'title'    => '',
				'callback' => 'html_section',
				'args'     => array(
					'option_name' => $option_name,
					'id'          => 'pro_notice_static_files',
					'class'       => 'wpo_wcpdf_pro_notice',
					'html'        => __( 'Attach up to 3 static files (for example, a terms & conditions document) to the WooCommerce emails of your choice.', 'woocommerce-pdf-invoices-packing-slips' ) . ' ' .
						'<a href="' . esc_url( admin_url( 'admin.php?page=wpo_wcpdf_options_page&tab=upgrade' ) ) . '">' . __( 'Upgrade today!', 'woocommerce-pdf-invoices-packing-slips' ) . '</a>'
				),
			);
		}

		// allow plugins to alter settings fields
		$settings_fields = apply_filters( 'wpo_wcpdf_settings_fields_general', $settings_fields, $page, $option_group, $option_name, $this );
		WPO_WCPDF()->settings->add_settings_fields( $settings_fields, $page, $option_group, $option_name );
	}

	public function attachment_settings_hint( $active_tab, $active_section ) {
		// save or check option to hide attachments settings hint
		if ( isset( $_REQUEST['wpo_wcpdf_hide_attachments_hint'] ) && isset( $_REQUEST['_wpnonce'] ) ) {
			// validate nonce
			if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ), 'hide_attachments_hint_nonce' ) ) {
				wcpdf_log_error( 'You do not have sufficient permissions to perform this action: wpo_wcpdf_hide_attachments_hint' );
				$hide_hint = false;
			} else {
				update_option( 'wpo_wcpdf_hide_attachments_hint', true );
				$hide_hint = true;
			}
		} else {
			$hide_hint = get_option( 'wpo_wcpdf_hide_attachments_hint' );
		}

		if ( $active_tab == 'general' && ! $hide_hint ) {
			$documents = WPO_WCPDF()->documents->get_documents();

			foreach ( $documents as $document ) {
				if ( $document->get_type() == 'invoice' ) {
					$invoice_email_ids = $document->get_attach_to_email_ids();
					if ( empty( $invoice_email_ids ) ) {
						include_once( WPO_WCPDF()->plugin_path() . '/views/attachment-settings-hint.php' );
					}
				}
			}
		}
	}

	/**
	 * Get the settings categories.
	 *
	 * @return array
	 */
	public function get_settings_categories(): array {
		$settings_categories = array(
			'display' => array(
				'title'   => __( 'Display Settings', 'woocommerce-pdf-invoices-packing-slips' ),
				'members' => array(
					'download_display',
					'paper_size',
					'template_path',
					'template_ink_saving',
					'test_mode',
				),
			),
			'shop_information' => array(
				'title'   => __( 'Shop Information', 'woocommerce-pdf-invoices-packing-slips' ),
				'members' => array(
					'header_logo',
					'header_logo_height',
					'shop_name',
					'shop_address_line_1',
					'shop_address_line_2',
					'shop_address_country',
					'shop_address_state',
					'shop_address_city',
					'shop_address_postcode',
					'shop_address_additional',
					'vat_number',
					'coc_number',
					'shop_phone_number',
					'shop_email_address',
					'footer',
				)
			),
			'advanced_formatting' => array(
				'title'   => __( 'Advanced Formatting', 'woocommerce-pdf-invoices-packing-slips' ),
				'members' => array(
					'font_subsetting',
					'currency_font',
					'extra_1',
					'extra_2',
					'extra_3',
				)
			),
			'checkout_field' => array(
				'title'   => __( 'Checkout Field', 'woocommerce-pdf-invoices-packing-slips' ),
				'members' => array(
					'checkout_field_enable',
					'checkout_field_label',
					'checkout_field_as_vat_number',
					'checkout_field_enable_my_account',
				)
			),
		);

		if ( ! function_exists( 'WPO_WCPDF_Pro' ) ) {
			$settings_categories['address_customization'] = array(
				'title'   => __( 'Customer Address Customization', 'woocommerce-pdf-invoices-packing-slips' ) . ' ' .
							 '<span class="wpo_wcpdf_badge badge-pro">' . __( 'Pro', 'woocommerce-pdf-invoices-packing-slips' ) . '</span>',
				'members' => array(
					'pro_notice_address_customization',
				),
			);

			$settings_categories['multilingual'] = array(
				'title'   => __( 'Multilingual', 'woocommerce-pdf-invoices-packing-slips' ) . ' ' .
							 '<span class="wpo_wcpdf_badge badge-pro">' . __( 'Pro', 'woocommerce-pdf-invoices-packing-slips' ) . '</span>',
				'members' => array(
					'pro_notice_multilingual',
				),
			);

			$settings_categories['static_files'] = array(
				'title'   => __( 'Static Files', 'woocommerce-pdf-invoices-packing-slips' ) . ' ' .
							 '<span class="wpo_wcpdf_badge badge-pro">' . __( 'Pro', 'woocommerce-pdf-invoices-packing-slips' ) . '</span>',
				'members' => array(
					'pro_notice_static_files',
				),
			);
		}

		return apply_filters( 'wpo_wcpdf_general_settings_categories', $settings_categories, $this );
	}

	/**
	 * List templates in plugin folder, theme folder & child theme folder
	 * @return array		template path => template name
	 */
	public function find_templates() {
		$installed_templates = array();
		// get base paths
		$template_base_path  = ( function_exists( 'WC' ) && is_callable( array( WC(), 'template_path' ) ) ) ? WC()->template_path() : apply_filters( 'woocommerce_template_path', 'woocommerce/' );
		$template_base_path  = untrailingslashit( $template_base_path );
		$template_paths      = array (
			// note the order: child-theme before theme, so that array_unique filters out parent doubles
			'default'     => WPO_WCPDF()->plugin_path() . '/templates/',
			'child-theme' => get_stylesheet_directory() . "/{$template_base_path}/pdf/",
			'theme'       => get_template_directory() . "/{$template_base_path}/pdf/",
		);

		$template_paths = apply_filters( 'wpo_wcpdf_template_paths', $template_paths );

		if ( defined( 'WP_CONTENT_DIR' ) && ! empty( WP_CONTENT_DIR ) && ! empty( ABSPATH ) && false !== strpos( WP_CONTENT_DIR, ABSPATH ) ) {
			$forwardslash_basepath = str_replace( '\\', '/', ABSPATH );
		} else {
			$forwardslash_basepath = str_replace( '\\', '/', WP_CONTENT_DIR );
		}

		foreach ( $template_paths as $template_source => $template_path ) {
			$dirs = (array) glob( $template_path . '*' , GLOB_ONLYDIR );

			foreach ( $dirs as $dir ) {
				if ( empty( $dir ) ) {
					continue;
				}
				// we're stripping abspath to make the plugin settings more portable
				$forwardslash_dir                      = str_replace( '\\', '/', $dir );
				$relative_path                         = ! empty( $forwardslash_dir ) ? str_replace( $forwardslash_basepath, '', $forwardslash_dir ) : '';
				$installed_templates[ $relative_path ] = basename( $dir );
			}
		}

		// remove parent doubles
		$installed_templates = array_unique( $installed_templates );

		if ( empty( $installed_templates ) && ! empty( $template_paths['default'] ) ) {
			// fallback to Simple template for servers with glob() disabled
			$simple_template_path = str_replace( ABSPATH, '', $template_paths['default'] . 'Simple' );
			$installed_templates[ $simple_template_path ] = 'Simple';
		}

		return apply_filters( 'wpo_wcpdf_templates', $installed_templates );
	}

	public function display_admin_notice_for_shop_address(): void {
		// Return if the notice has been dismissed.
		if ( get_option( 'wpo_wcpdf_dismiss_shop_address_notice', false ) ) {
			return;
		}

		// Handle dismissal action.
		if ( isset( $_GET['wpo_dismiss_shop_address_notice'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( isset( $_GET['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'dismiss_shop_address_notice' ) ) {
				update_option( 'wpo_wcpdf_dismiss_shop_address_notice', true );
				wp_safe_redirect( remove_query_arg( array( 'wpo_dismiss_shop_address_notice', '_wpnonce' ) ) );
				exit;
			} else {
				wcpdf_log_error( 'You do not have sufficient permissions to perform this action: wpo_dismiss_requirements_notice' );
				return;
			}
		}

		$general_settings = WPO_WCPDF()->settings->general;
		$display_notice   = false;
		$languages_data   = wpo_wcpdf_get_multilingual_languages();
		$languages        = $languages_data ? array_keys( $languages_data ) : array( 'default' );

		foreach ( $languages as $language ) {
			$line_1   = $general_settings->get_setting( 'shop_address_line_1', $language ) ?? '';
			$country  = $general_settings->get_setting( 'shop_address_country', $language ) ?? '';
			$states   = wpo_wcpdf_get_country_states( $country );
			$state    = ! empty( $states ) ? $general_settings->get_setting( 'shop_address_state', $language ) : '';
			$city     = $general_settings->get_setting( 'shop_address_city', $language ) ?? '';
			$postcode = $general_settings->get_setting( 'shop_address_postcode', $language ) ?? '';

			if (
				empty( $line_1 ) ||
				empty( $country ) ||
				( ! empty( $states ) && empty( $state ) ) || // only require state if states exist
				empty( $city ) ||
				empty( $postcode )
			) {
				$display_notice = true;
				break;
			}
		}

		if ( $display_notice ) {
			$general_page_url = admin_url( 'admin.php?page=wpo_wcpdf_options_page&tab=general' );
			$dismiss_url      = wp_nonce_url( add_query_arg( 'wpo_dismiss_shop_address_notice', true ), 'dismiss_shop_address_notice' );
			$notice_message   = sprintf(
				/* translators: 1: Plugin name, 2: Open anchor tag, 3: Close anchor tag */
				__( '%1$s: Your shop address is incomplete. Please fill in the missing fields in the %2$sGeneral settings%3$s.', 'woocommerce-pdf-invoices-packing-slips' ),
				'<strong>PDF Invoices & Packing Slips for WooCommerce</strong>',
				'<a href="' . esc_url( $general_page_url ) . '">',
				'</a>'
			);

			?>

			<div class="notice notice-warning">
				<p><?php echo wp_kses_post( $notice_message ); ?></p>
				<p><a href="<?php echo esc_url( $dismiss_url ); ?>"
					  class="wpo-wcpdf-dismiss"><?php esc_html_e( 'Hide this message', 'woocommerce-pdf-invoices-packing-slips' ); ?></a>
				</p>
			</div>

			<?php
		}

	}

	/**
	 * Get the states for a given country code via AJAX.
	 */
	public function ajax_get_shop_country_states() {
		// Accept either the settings nonce or the setup-wizard nonce.
		$valid = check_ajax_referer( 'wpo_wcpdf_admin_nonce', 'security', false )
			|| check_ajax_referer( 'wpo_wcpdf_setup_nonce', 'security', false );

		if ( ! $valid ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce.', 'woocommerce-pdf-invoices-packing-slips' ) ), 403 );
		}
		
		$request = stripslashes_deep( $_POST );

		if ( empty( $request['country'] ) ) {
			wp_send_json_error( array( 'message' => __( 'No country code provided.', 'woocommerce-pdf-invoices-packing-slips' ) ) );
		}

		$country_code = sanitize_text_field( $request['country'] );
		$states       = wpo_wcpdf_get_country_states( $country_code );
		$selected     = strtoupper( trim( $this->get_setting( 'shop_address_state' ) ) );

		wp_send_json_success(
			array(
				'states'   => $states ?: array(),
				'selected' => $selected ?: '',
			)
		);
	}

	/**
	 * Get a general setting key value, optionally using a locale-specific sub-key.
	 *
	 * @param string $key     The key of the setting to retrieve.
	 * @param string $locale  Optional. Locale to retrieve. Falls back to 'default' if not provided or not found.
	 * @return string The value of the setting.
	 */
	public function get_setting( string $key, string $locale = '' ): string {
		if ( empty( $key ) ) {
			return '';
		}

		$general_settings = get_option( $this->option_name, array() );
		$setting_text     = '';

		if ( ! empty( $general_settings[ $key ] ) ) {
			$setting = $general_settings[ $key ];

			if ( is_array( $setting ) ) {
				if ( ! empty( $locale ) && array_key_exists( $locale, $setting ) ) {
					$setting_text = $setting[ $locale ];
				} elseif ( array_key_exists( 'default', $setting ) ) {
					$setting_text = $setting['default'];
				} else {
					$setting_text = reset( $setting );
				}
			} else {
				$setting_text = $setting;
			}
		}

		return apply_filters(
			'wpo_wcpdf_get_general_setting',
			wptexturize( trim( $setting_text ) ),
			$key,
			$locale,
			$general_settings
		);
	}
	
	/**
	 * Collect documents whose template files are missing.
	 *
	 * @return string[] Array of document titles.
	 */
	private function get_missing_template_files(): array {
		$template_path       = WPO_WCPDF()->settings->get_template_path();
		$template_path_array = explode( '/', $template_path );
		$template_name       = end( $template_path_array );
		$enabled_documents   = WPO_WCPDF()->documents->get_documents( 'enabled' );
		$missing             = array();

		foreach ( $enabled_documents as $doc ) {
			$filename         = $doc->get_type() . '.php';
			$located_template = $doc->locate_template_file( $filename );

			// If using Simple OR the located template is not inside /Simple/, and the file exists, skip.
			if (
				( 'Simple' === $template_name || false === strpos( $located_template, '/Simple/' ) ) &&
				WPO_WCPDF()->file_system->exists( $located_template )
			) {
				continue;
			}

			$missing[] = $doc->get_title();
		}

		return $missing;
	}

	/**
	 * Build the HTML notice for missing template files.
	 *
	 * @param string[] $missing_titles
	 * @return string
	 */
	private function render_missing_template_files_notice( array $missing_titles ): string {
		if ( empty( $missing_titles ) ) {
			return '';
		}

		$missing_list = wp_sprintf_l( '%l', array_map( 'esc_html', $missing_titles ) );
		$file_label   = _n( 'a file', 'files', count( $missing_titles ), 'woocommerce-pdf-invoices-packing-slips' );

		$notice  = '<div class="notice notice-info inline notice-wpo"><p>';
		$notice .= sprintf(
			/* translators: 1: "file"/"files" label (singular/plural), 2: list of document titles */
			esc_html__( 'Your custom template folder does not contain %1$s for: %2$s.', 'woocommerce-pdf-invoices-packing-slips' ),
			esc_html( $file_label ),
			' <strong>' . $missing_list . '</strong>'
		);

		$notice .= ' ' . sprintf(
			/* translators: 1: Opening <strong>, 2: Closing </strong> */
			esc_html__( 'Documents where files are missing will remain functional and will use the default %1$sSimple%2$s template.', 'woocommerce-pdf-invoices-packing-slips' ),
			'<strong>',
			'</strong>'
		);

		// Premium Templates guidance (only if bundle license missing/invalid)
		if ( function_exists( 'WPO_WCPDF_Templates' ) ) {
			$license_info = WPO_WCPDF()->settings->upgrade->get_extension_license_infos();
			$info         = $license_info['bundle'] ?? null;

			if ( empty( $info['status'] ) || 'valid' !== $info['status'] ) {
				$template_folder = str_replace( wp_normalize_path( ABSPATH ), '', wp_normalize_path( WPO_WCPDF_Templates()->plugin_path() ) . '/templates/...' );

				$notice .= '<br><br>';
				$notice .= sprintf(
					/* translators: %s: Template folder path */
					esc_html__( 'If you are not using the latest version of the Premium Templates extension, please update it. Otherwise, copy the template files from the newest version in %s and adapt them to your custom template.', 'woocommerce-pdf-invoices-packing-slips' ),
					'<code>' . esc_html( $template_folder ) . '</code>'
				);
			}
		}

		$notice .= '<br><br>';
		$notice .= sprintf(
			/* translators: 1: opening <a>, 2: closing </a> */
			esc_html__( 'Please refer to the %1$sCreating a custom PDF template%2$s article to learn how to update your template.', 'woocommerce-pdf-invoices-packing-slips' ),
			'<a href="' . esc_url( 'https://docs.wpovernight.com/woocommerce-pdf-invoices-packing-slips/creating-a-custom-pdf-template/' ) . '" target="_blank" rel="noopener noreferrer">',
			'</a>'
		);

		$notice .= '</p></div>';

		return $notice;
	}

}

endif; // class_exists
